/* Debounce a button input, and create a pulse exactly 1 'clk' clock in length.
 */

`define STATE_ZERO 0
`define STATE_ONE  1

`define A_FEW  8'hff

module debouncer(clk, reset, buttonin, outpulse);
   input clk;
   input reset;
   input buttonin;
   output outpulse;

   /* Method of operation:
      We take the asynchronous button input, and pipeline that through two FFs
      as a 'synchroniser' (look this up).
    
      The resulting signal is in our clock domain; it may still be 'bouncy' though.

      We have a toy state machine here that counts samples of this signal.  If it
      sees a few that are all the same, it deems the output to be in an 0 or 1 state.
    */
   reg synch_1, synch_2;
   reg [7:0] count;
   wire      outpulse;
   reg       state;
   reg       last_state;

   always @ (posedge clk or posedge reset)
     begin
        if (reset)
          begin
             synch_1 <= 0;
             synch_2 <= 0;
             state <= `STATE_ZERO;
             count <= 0;
          end
        else
          begin
             /* <= is non-blocking assignment (as opposed to =, blocking).
                It's most familiar to me from the schematic side of things; in this
                pipeline, synch_1 gets the new value and synch_2 gets what synch_1 was.
                Draw a FF whose output is feeding into a second FF, and you'll see.
              */
             synch_1 <= buttonin;
             synch_2 <= synch_1;

             case (state)
               `STATE_ZERO:
                 begin
                    if (synch_2) /* Input is 1 */
                      begin
                         if (count == `A_FEW)
                           begin
                              /* We saw a few ones-- switch state */
                              state <= `STATE_ONE;
                              count <= 0;
                           end
                         else
                           begin
                              count <= count + 1;
                           end
                      end
                    else /* Input's 0 */
                      begin
                         /* Even if we saw five ones, we've seen a zero; the input hasn't
                          settled yet, so start counting again. */
                         count <= 0;
                      end // else: !if(count == `A_FEW)
                 end
               `STATE_ONE:
                 begin
                    if (!synch_2) /* Input is 0 */
                      begin
                         if (count == `A_FEW)
                           begin
                              /* We saw a few ones-- switch state */
                              state <= `STATE_ZERO;
                              count <= 0;
                           end
                         else
                           begin
                              count <= count + 1;
                           end
                      end
                    else /* Input's 1 */
                      begin
                         /* Even if we saw five ones, we've seen a zero; the input hasn't
                          settled yet, so start counting again. */
                         count <= 0;
                      end // else: !if(count == `A_FEW)
                 end
             endcase // case (state)

             last_state <= state; /* See assign below */
          end
     end

   /* This condition is true when we see an 0 -> 1 transition, and only lasts for
      1 pulse.
    */
   assign outpulse = (!last_state & state);

endmodule
