module top_level(clk0, button0,
                 digit_en,
                 segments);
   input clk0;
   input button0;
   output [3:0] digit_en;
   output [7:0] segments; /* Includes decimal point */

   /************************** Misc clock/reset **************************/

   wire clk;
   wire reset;

   /* Clock input from external oscillator must go through an IBUFG to
      drive the internal clock network. */
   IBUFG cb0 (.I(clk0), .O(clk));
   reset_gen myresetter(.clk(clk), .reset(reset));

   /************************** Counters **********************************/

   wire [3:0] count_h;
   wire [3:0] count_l;
   wire [6:0] seg_h;
   wire [6:0] seg_l;
   wire       ctr_en;
   wire       carryout;

   /* Wire the counters, with carry & enable */
   counter ctr_h(.clk(clk), .reset(reset),
		 .enable(carryout & ctr_en), .count(count_h));
   counter ctr_l(.clk(clk), .reset(reset),
		 .enable(ctr_en), .count(count_l), .carry_out(carryout));

   /************************** 7-seg decoders ****************************/

   /* The decoders take the count_* buses and produce segments */
   bcd_to_7seg ssdec_h(.bcd(count_h), .segments(seg_h));
   bcd_to_7seg ssdec_l(.bcd(count_l), .segments(seg_l));

   /************************** Button input ******************************/

   /* Finally, this provides an enable pulse (one clock long) to get our counters
      to count.
    */
   debouncer deb(.clk(clk), .reset(reset), .buttonin(button0), .outpulse(ctr_en));

   /************************** LED display output ************************/

   /* Multiplexed 7-segment LED display:
      We have a 50MHz system clock.  Implement another counter that counts to
      Some Large Number, and flip the digit being displayed each time this
      rolls over.  We choose 0x40000, which means we flip at approx 1/190.73Hz.
    */
   reg [17:0] disp_counter;
   reg        digit_displayed;

   always @ (posedge clk or posedge reset)
     begin
        if (reset)
          begin
             disp_counter <= 0;
             digit_displayed <= 0;
          end
        else
          if (disp_counter != 18'h3ffff)
            begin
               disp_counter <= disp_counter + 1;
            end
          else
            begin
               disp_counter <= 0;
               digit_displayed <= !digit_displayed; /* Display other digit */
            end
     end

   /* Segments are actually '0' for 'on', so invert here */
   assign segments[7] = 1; /* DP off */
   assign digit_en[3:2] = 2'b11;
   /* This does the flipping:  A digit is enabled and the correct segment data
      multiplexed into the common segments output bus.
    */
   assign segments[6:0] = digit_displayed ? ~seg_l : ~seg_h;
   assign digit_en[1:0] = {digit_displayed, !digit_displayed};

endmodule
                 